/**
* \file: sdc_access.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <sdc_op_conv.h>

#include "encryption/sdc_access.h"
#include "util/logger.h"
#include "util/helper.h"

typedef enum sdc_unwrap_stages
{
	SDC_EXTRACT_TYPE,
	SDC_OPEN_SESSION,
	SDC_LOAD_KEY,
	SDC_UNWRAP,
	SDC_CLOSE_SESSION,
	SDC_FREE_TYPE,
	SDC_UNWRAPPING_DONE
} sdc_unwrap_stages;

typedef enum sdc_wrap_stages
{
	SDC_WRAP_OPEN_SESSION,
	SDC_WRAP_LOAD_KEY,
	SDC_WRAP_BUF,
	SDC_WRAP_CLOSE_SESSION,
	SDC_WRAPPING_DONE
} sdc_wrapping_stages;

typedef struct sdc_data {
    unsigned char *data;
    size_t len;
} sdc_data;

/* Filename extension for wrapped files */
#define WRAP_EXTENSION ".wrap"
/* ID of the key used for wrapping */
#define KEY_ID 1

static error_code_t sdc_access_unwrap_file(const unsigned char *wrapped_data,
		size_t wrapped_data_len, unsigned char **unwrapped_data, size_t* unwrapped_data_len);

static error_code_t sdc_access_unwrap_file(const unsigned char *wrapped_data,
		size_t wrapped_data_len, unsigned char **unwrapped_data, size_t* unwrapped_data_len)
{
	sdc_session_t *session = NULL;
	sdc_wrap_unwrap_type_t *wrap_unwrap_type = NULL;
	sdc_unwrap_stages stage = SDC_EXTRACT_TYPE;
	sdc_error_t sdc_err = SDC_OK;
	error_code_t err = RESULT_OK;

	while(stage < SDC_UNWRAPPING_DONE )
		{
			switch(stage)
			{
				case SDC_EXTRACT_TYPE:
					sdc_err = sdc_unwrap_formatted_extract_type(wrapped_data,
							wrapped_data_len, &wrap_unwrap_type);
					if (SDC_OK != sdc_err)
					{
						stage = SDC_UNWRAPPING_DONE;
						logger_log_error("Could not extract type from the wrapped file: %d", sdc_err);
					}
					break;
				case SDC_OPEN_SESSION:
					sdc_err = sdc_open_session(&session);
					if (SDC_OK != sdc_err)
					{
						stage = SDC_FREE_TYPE;
						logger_log_error("Could not create SDC session for unwrapping: %d", sdc_err);
					}
					break;
				case SDC_LOAD_KEY:
					sdc_err = sdc_session_load_storage_key(session, KEY_ID);
					if (SDC_OK != sdc_err)
					{
						stage = SDC_CLOSE_SESSION;
						logger_log_error("Could not set SDC session key for unwrapping: %d", sdc_err);
					}
					break;
				case SDC_UNWRAP:
					sdc_err = sdc_unwrap_formatted(session, wrap_unwrap_type,
								wrapped_data, wrapped_data_len,
								unwrapped_data, unwrapped_data_len);
					if (SDC_OK != sdc_err)
					{
						stage = SDC_CLOSE_SESSION;
						logger_log_error("Could not unwrap file: %d", sdc_err);
					}
					break;
				case SDC_CLOSE_SESSION:
					sdc_err = sdc_close_session(session);
					if (SDC_OK != sdc_err)
					{
						stage = SDC_FREE_TYPE;
						logger_log_error("Could not close the session after unwrapping: %d", sdc_err);
					}
					break;
				case SDC_FREE_TYPE:
					if (wrap_unwrap_type) {
						sdc_err = sdc_wrap_unwrap_type_free(wrap_unwrap_type);
						if (SDC_OK != sdc_err )
						{
							stage = SDC_UNWRAPPING_DONE;
							logger_log_error("Could not free the type after unwrapping: %d", sdc_err);
						}
					}
					break;
				default:
					/*Nothing to do*/
					break;
			}

			if (SDC_OK == sdc_err)
				stage ++;
			else
				err = RESULT_CRYPTOGRAPHIC_OPERATION_FAILED;

		}
	return err;
}

error_code_t sdc_access_load_state_file(const char *fn, void *buf, size_t buf_size)
{
	FILE *fd=NULL;
	sdc_data wrapped_data = {NULL, 0};
	sdc_data unwrapped_data = {NULL, 0};
	char *wrap_fn;
	size_t read_bytes;
	long file_size=0;
	error_code_t err = RESULT_OK;

	wrap_fn = malloc(strlen(fn) + sizeof(WRAP_EXTENSION));
	if (NULL == wrap_fn)
	{
		logger_log_error("Failed to allocate memory for filename");
		err = RESULT_NORESOURCE;
	}
	else
	{
		strcpy(wrap_fn, fn);
		strncat(wrap_fn, WRAP_EXTENSION,sizeof(WRAP_EXTENSION));

		fd=fopen(wrap_fn,"r");
		if (NULL == fd)
		{
			logger_log_error("Failed to open state file: %s - %s", wrap_fn,
					strerror(errno));
			err = RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
		}
		else
		{
			fseek(fd, 0, SEEK_END);
			file_size = ftell(fd);
			if (file_size <= 0)
			{
				logger_log_error("Failed to read state file size: %s - %s", wrap_fn,
						strerror(errno));
				err = RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
			}
			else
			{
				wrapped_data.len = file_size;
				wrapped_data.data = malloc(wrapped_data.len);
				if (wrapped_data.data == NULL)
				{
					logger_log_error("Failed to allocate memory to read "
							"state file");
					err = RESULT_NORESOURCE;
				}
				else
				{
					fseek(fd, 0, SEEK_SET);
					read_bytes = fread(wrapped_data.data, 1, wrapped_data.len, fd);
					if (read_bytes < (size_t) wrapped_data.len)
					{
						logger_log_error("Could not read state file: Read %d To"
								" be read %d : %s", read_bytes, wrapped_data.len
								, strerror(errno));
						err = RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
					}
					else
					{
						err = sdc_access_unwrap_file(wrapped_data.data,
								wrapped_data.len, &unwrapped_data.data, &unwrapped_data.len);
						if (RESULT_OK == err)
						{
							if (unwrapped_data.len != buf_size)
							{
								logger_log_error("Invalid size of unpacked state "
										"file: %d != %d",unwrapped_data.len, buf_size);
									err = RESULT_CRYPTOGRAPHIC_OPERATION_FAILED;
							}
							else
							{
									memcpy(buf, unwrapped_data.data, buf_size);
							}
						}
					}
				}
			}
		}
	}

	if (unwrapped_data.data)
		free(unwrapped_data.data);
	if(wrapped_data.data)
		free(wrapped_data.data);
	if(wrap_fn)
		free(wrap_fn);
	if(fd)
	fclose(fd);

	return err;
}

error_code_t sdc_access_store_state_file(const char *fn, const void *buf, size_t buf_size)
{
	FILE *fd = NULL;
	int fd_ival;
	char *wrap_fn = NULL;
	size_t written = 0;
	const sdc_wrap_unwrap_type_t *wrap_unwrap_type = NULL;
	sdc_wrapping_stages stage = SDC_WRAP_OPEN_SESSION;
	sdc_data wrapped_data = {NULL, 0};
	sdc_session_t *session = NULL;
	sdc_error_t sdc_err = SDC_OK;
	error_code_t err = RESULT_OK;

	while(stage < SDC_WRAPPING_DONE )
	{
		switch(stage)
		{
			case SDC_WRAP_OPEN_SESSION:
				sdc_err = sdc_open_session(&session);
				if (sdc_err != SDC_OK)
				{
					logger_log_error("Could not create SDC session: %d", sdc_err);
					stage = SDC_WRAPPING_DONE;
				}
				break;
			case SDC_WRAP_LOAD_KEY:
				sdc_err = sdc_session_load_storage_key(session, KEY_ID);
				if (sdc_err != SDC_OK)
				{
					logger_log_error("Could not set SDC session key: %d", sdc_err);
					stage = SDC_WRAP_CLOSE_SESSION;
				}
				break;
			case SDC_WRAP_BUF:
				wrap_unwrap_type = (sdc_wrap_unwrap_type_t*)sdc_wrap_unwrap_get_default();
				sdc_err = sdc_wrap_formatted(session, wrap_unwrap_type, (uint8_t *) buf,
							buf_size, (uint8_t **) &wrapped_data.data, &wrapped_data.len);
				if (sdc_err != SDC_OK)
				{
					logger_log_error("Could not wrap state file: %d", sdc_err);
					stage = SDC_WRAP_CLOSE_SESSION;
				}
				break;
			case SDC_WRAP_CLOSE_SESSION:
				sdc_err = sdc_close_session(session);
				if (SDC_OK != sdc_err)
				{
					stage = SDC_WRAPPING_DONE;
					logger_log_error("Could not close the session after unwrapping: %d", sdc_err);
				}
				break;
			default:
				/*Nothing to do*/
				break;
		}
		if(SDC_OK == sdc_err)
			stage++;
		else
			err = RESULT_CRYPTOGRAPHIC_OPERATION_FAILED;
	}
	if (RESULT_OK == err)
	{
		wrap_fn = malloc(strlen(fn) + sizeof(WRAP_EXTENSION));
		if (wrap_fn == NULL)
		{
			logger_log_error("Failed to allocate memory for filename");
			err =  RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
		}
		else
		{
			strcpy(wrap_fn, fn);
			strncat(wrap_fn, WRAP_EXTENSION, sizeof(WRAP_EXTENSION));

			fd=fopen(wrap_fn,"w+");
			if (fd == NULL)
			{
				logger_log_error("Failed to open state file for writing: %s - %s",
						wrap_fn, strerror(errno));
				err =  RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
			}
			else
			{
				written = fwrite(wrapped_data.data, 1, wrapped_data.len, fd);
				if (written < wrapped_data.len)
				{
					logger_log_error("Could not write state file: Read %d To be read %d : %s",
							read, wrapped_data.len, strerror(errno));
					err =  RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
				}

				if (RESULT_OK == err)
				{
					if(fflush(fd) != 0)
					{
						logger_log_error("Failed to flush data into state file : %s - %s",
								wrap_fn, strerror(errno));
						err =  RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
					}
					else
					{
						fd_ival=fileno(fd);
						if (fd_ival < 0)
						{
							logger_log_error("Failed to get state file info to sync: %s - %s",
									wrap_fn, strerror(errno));
							err =  RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
						}
						else
						{
							if (fsync(fd_ival) < 0)
							{
								logger_log_error("Failed to sync state file : %s - %s",
										wrap_fn, strerror(errno));
								err =  RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
							}
						}
					}
				}
			}
			/* In case the state file is not existing before it will be created by the above fopen
			 *    Sync parent directory in order to ensure this is reflected in the file-system
			 *    In case the sync fails we can ignore it, as a missing state file would re-trigger this
			 *    code during next reboot */
			helper_sync_parent_directory(wrap_fn);
		}
	}



	if(wrap_fn)
		free(wrap_fn);
	if(wrapped_data.data)
		free(wrapped_data.data);
	if(fd)
		fclose(fd);

	return err;
}

error_code_t sdc_access_decrypt_memory_block(char **decr_buffer_ptr, size_t *decr_buf_size,
		const char *encr_buf, size_t encr_buf_size)
{

	return sdc_access_unwrap_file((const unsigned char *)encr_buf, encr_buf_size,
			(unsigned char **)decr_buffer_ptr, decr_buf_size);

}

void sdc_access_free_buffer(char *key_buffer)
{
	free(key_buffer);
}
